<?php

namespace App\Http\Controllers\Karyawan;

use Exception;
use Carbon\Carbon;
use App\Models\Cuti;
use App\Models\JatahCuti;
use App\Models\JenisCuti;
use App\Models\Notifikasi;
use App\Models\CutiApproval;
use App\Models\AjukanShift;
use App\Models\JadwalShift;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class PengajuanController extends Controller
{
    /**
     * Display unified index page with both cuti and shift data
     */
    public function index()
    {
        $user = Auth::user();
        $karyawan = $user->karyawan;

        // ===== DATA CUTI =====
        // Get jatah cuti tahun ini
        $jatahCuti = JatahCuti::where('karyawan_id', $karyawan->id)
            ->where('tahun', now()->year)
            ->first();

        // Get pengajuan cuti yang masih aktif (pending)
        $cutiAktif = Cuti::with(['jenisCuti', 'approvals.approver'])
            ->where('karyawan_id', $karyawan->id)
            ->where('status', 'pending')
            ->latest()
            ->get();

        // Get pengajuan cuti yang sudah selesai (disetujui/ditolak)
        $cutiSelesai = Cuti::with(['jenisCuti', 'approvals.approver'])
            ->where('karyawan_id', $karyawan->id)
            ->whereIn('status', ['disetujui', 'ditolak'])
            ->latest()
            ->get();

        // ===== DATA SHIFT (Only for Admin Roles) =====
        $jadwalShiftAktif = null;
        $shiftAktif = collect();
        $shiftSelesai = collect();

        $adminRoles = ['admin', 'manager', 'gm', 'hrd'];
        $isAdmin = in_array($user->role, $adminRoles);

        if ($isAdmin && $karyawan->departemen_id) {
            $departemen = $karyawan->departemen;

            // Shift aktif departemen
            $jadwalShiftAktif = JadwalShift::where('departemen_id', $departemen->id)
                ->where('is_active', true)
                ->with('shift')
                ->first();

            // Pengajuan shift yang masih aktif (pending)
            $shiftAktif = AjukanShift::with(['shiftLama', 'shiftBaru', 'pemohon', 'approver'])
                ->where('departemen_id', $departemen->id)
                ->where('requested_by', $user->id)
                ->where('status', 'pending')
                ->latest()
                ->get();

            // Pengajuan shift yang sudah selesai
            $shiftSelesai = AjukanShift::with(['shiftLama', 'shiftBaru', 'pemohon', 'approver'])
                ->where('departemen_id', $departemen->id)
                ->where('requested_by', $user->id)
                ->whereIn('status', ['disetujui', 'ditolak'])
                ->latest()
                ->get();
        }

        return view('karyawan.pengajuan.index', compact(
            // Cuti data
            'jatahCuti',
            'cutiAktif',
            'cutiSelesai',
            // Shift data
            'jadwalShiftAktif',
            'shiftAktif',
            'shiftSelesai'
        ));
    }

    /**
     * Show create form for cuti
     */
    public function create()
    {
        // Get jenis cuti yang aktif
        $jenisCuti = JenisCuti::all();

        return view('karyawan.pengajuan.create', compact('jenisCuti'));
    }

    /**
     * Store new cuti
     */
    public function storeCuti(Request $request)
    {
        try {
            $request->validate([
                'jenis_id' => 'required|exists:jenis_cuti,id',
                'tanggal_mulai' => 'required|date|after_or_equal:today',
                'tanggal_selesai' => 'required|date|after_or_equal:tanggal_mulai',
                'alasan' => 'required|string|min:10|max:500',
                'file_pendukung' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048'
            ]);

            $user = Auth::user();
            $karyawan = $user->karyawan;

            // Validasi jenis cuti
            $jenisCuti = JenisCuti::findOrFail($request->jenis_id);

            // Hitung jumlah hari
            $tanggalMulai = Carbon::parse($request->tanggal_mulai);
            $tanggalSelesai = Carbon::parse($request->tanggal_selesai);
            $jumlahHari = $tanggalMulai->diffInDays($tanggalSelesai) + 1;

            // Validasi max hari untuk jenis cuti ini
            if ($jenisCuti->max_hari && $jumlahHari > $jenisCuti->max_hari) {
                return back()->with('error', "Maksimal {$jenisCuti->nama} adalah {$jenisCuti->max_hari} hari");
            }

            // Cek jatah cuti
            $jatah = JatahCuti::where('karyawan_id', $karyawan->id)
                ->where('tahun', now()->year)
                ->first();

            if (!$jatah || $jatah->jatah < $jumlahHari) {
                return back()->with('error', 'Sisa jatah cuti Anda tidak mencukupi');
            }

            // Cek bentrok dengan cuti lain
            $bentrok = Cuti::where('karyawan_id', $karyawan->id)
                ->where('status', '!=', 'ditolak')
                ->where(function($query) use ($request) {
                    $query->whereBetween('tanggal_mulai', [$request->tanggal_mulai, $request->tanggal_selesai])
                          ->orWhereBetween('tanggal_selesai', [$request->tanggal_mulai, $request->tanggal_selesai])
                          ->orWhere(function($q) use ($request) {
                              $q->where('tanggal_mulai', '<=', $request->tanggal_mulai)
                                ->where('tanggal_selesai', '>=', $request->tanggal_selesai);
                          });
                })
                ->exists();

            if ($bentrok) {
                return back()->with('error', 'Tanggal cuti bentrok dengan pengajuan cuti lain');
            }

            // Handle file upload
            $filePath = null;
            if ($request->hasFile('file_pendukung')) {
                $file = $request->file('file_pendukung');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = $file->storeAs('cuti-files', $fileName, 'public');
            }

            DB::beginTransaction();

            // Tentukan approval workflow berdasarkan role pengaju
            $approvalSteps = Cuti::getApprovalSteps($user->role);
            $firstStep = $approvalSteps[0] ?? null;

            // Buat data cuti
            $cuti = Cuti::create([
                'karyawan_id' => $karyawan->id,
                'jenis_id' => $request->jenis_id,
                'tanggal_mulai' => $request->tanggal_mulai,
                'tanggal_selesai' => $request->tanggal_selesai,
                'alasan' => $request->alasan,
                'file_pendukung' => $filePath,
                'status' => empty($approvalSteps) ? 'disetujui' : 'pending',
                'current_step' => $firstStep,
                'is_bentrok' => false
            ]);

            // Buat approval steps jika ada
            if (!empty($approvalSteps)) {
                foreach ($approvalSteps as $step) {
                    CutiApproval::create([
                        'cuti_id' => $cuti->id,
                        'step' => $step,
                        'status' => 'pending'
                    ]);
                }
            } else {
                // Jika tidak ada approval (HRD/Super Admin), langsung kurangi jatah
                $jatah->decrement('jatah', $jumlahHari);
            }

            // Activity log
            activity_log(
                'cuti',
                'create',
                "Mengajukan cuti {$jenisCuti->nama} tanggal {$request->tanggal_mulai} s/d {$request->tanggal_selesai}"
            );

            // Notifikasi untuk pemohon
            Notifikasi::create([
                'user_id' => $user->id,
                'judul' => 'Pengajuan Cuti Berhasil',
                'pesan' => "Pengajuan {$jenisCuti->nama} Anda berhasil dikirim dan menunggu persetujuan.",
                'type' => 'cuti',
                'target_role' => 'karyawan'
            ]);

            // Notifikasi untuk approver pertama
            if ($firstStep) {
                $roleLabel = ucfirst($firstStep);
                Notifikasi::create([
                    'user_id' => null,
                    'judul' => 'Pengajuan Cuti Baru',
                    'pesan' => "{$karyawan->user->nama} mengajukan {$jenisCuti->nama} ({$jumlahHari} hari) yang memerlukan persetujuan {$roleLabel}.",
                    'type' => 'cuti',
                    'target_role' => $firstStep
                ]);
            }

            DB::commit();

            return redirect()->route('karyawan.pengajuan.index')
                ->with('success', 'Pengajuan cuti berhasil dikirim');

        } catch (Exception $e) {
            DB::rollBack();

            // Delete uploaded file if exists
            if (isset($filePath)) {
                Storage::disk('public')->delete($filePath);
            }

            Log::error('Error Store Cuti: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Show riwayat page
     */
    public function riwayat()
    {
        $karyawan = Auth::user()->karyawan;

        $cuti = Cuti::with(['jenisCuti', 'approvals.approver'])
            ->where('karyawan_id', $karyawan->id)
            ->latest()
            ->paginate(20);

        return view('karyawan.pengajuan.riwayat', compact('cuti'));
    }

    /**
     * Show detail cuti
     */
    public function showCuti($id)
    {
        $cuti = Cuti::with([
                'karyawan.user',
                'jenisCuti',
                'approvals.approver'
            ])
            ->findOrFail($id);

        // Pastikan hanya pemilik cuti yang bisa lihat detail
        if ($cuti->karyawan->user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        return view('karyawan.pengajuan.detail', compact('cuti'));
    }

    /**
     * Cancel cuti
     */
    public function cancelCuti($id)
    {
        try {
            $cuti = Cuti::findOrFail($id);

            // Hanya pemilik cuti yang bisa cancel
            if ($cuti->karyawan->user_id !== Auth::id()) {
                return back()->with('error', 'Anda tidak memiliki akses');
            }

            // Hanya cuti pending yang bisa dicancel
            if ($cuti->status !== 'pending') {
                return back()->with('error', 'Hanya cuti yang sedang pending yang dapat dibatalkan');
            }

            DB::transaction(function () use ($cuti) {
                $cuti->update([
                    'status' => 'ditolak',
                    'catatan_admin' => 'Dibatalkan oleh pemohon'
                ]);

                // Update semua approval menjadi ditolak
                $cuti->approvals()->update(['status' => 'ditolak']);

                activity_log(
                    'cuti',
                    'cancel',
                    "Membatalkan pengajuan cuti tanggal {$cuti->tanggal_mulai->format('d/m/Y')}"
                );
            });

            return back()->with('success', 'Pengajuan cuti berhasil dibatalkan');

        } catch (Exception $e) {
            Log::error('Error Cancel Cuti: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }
}
